# Function tests a Null Hypothesis based on a null value and ROPE based on a posterior MCMC sample of a parameter
# Author: Michail Tsikerdekis
# compVal is the null value
# ropeRad is the radius
# HDImass is the confidence interval for HDI
# For more info on the method see: Kruschke, J.K. (2013). Bayesian estimation supersedes the t test. Journal of Experimental Psychology, 142(2), 573-603. doi: 10.1037/a0029146

H0testUsingROPE <- function(paramSampleVec, compVal = 0, ropeRad = 0.25, HDImass = .95){
meansample = mean(paramSampleVec)
mediansample = median(paramSampleVec)
mcmcDensity = density(paramSampleVec)
modesample = mcmcDensity$x[which.max(mcmcDensity$y)]

# Calculate HDI based on credMass
sortedPts = sort( paramSampleVec )
ciIdxInc = floor( HDImass * length( sortedPts ) )
nCIs = length( sortedPts ) - ciIdxInc
ciWidth = rep( 0 , nCIs )
for ( i in 1:nCIs ) {
  ciWidth[ i ] = sortedPts[ i + ciIdxInc ] - sortedPts[ i ]
}
HDImin = sortedPts[ which.min( ciWidth ) ]
HDImax = sortedPts[ which.min( ciWidth ) + ciIdxInc ]
HDIlim = c( HDImin , HDImax )

histinfo = hist( paramSampleVec , plot=F )
cenTendHt = 0.9*max(histinfo$density)
cvHt = 0.7*max(histinfo$density)
ROPEtextHt = 0.55*max(histinfo$density)

# Number of values (percentage) above the comparing value
pcgtCompVal = round( 100 * sum( paramSampleVec > compVal ) / length( paramSampleVec )  , 1 ) 
# Number of values in percentage below the point
pcltCompVal = 100 - pcgtCompVal 

ROPE = compVal+c(-ropeRad,ropeRad)
pcInROPE =  (sum( paramSampleVec > ROPE[1] & paramSampleVec < ROPE[2] ) / length( paramSampleVec ) )

HDISample = paramSampleVec[paramSampleVec >= HDImin & paramSampleVec <= HDImax]
HDIinROPE = ( sum( HDISample > ROPE[1] & HDISample < ROPE[2] ) / length( HDISample ) )

results <- matrix(c(round(meansample,3),round(mediansample,3),round(modesample,3),round(HDImin,3),round(HDImax,3),compVal,paste(round(pcgtCompVal,3),"%",sep=""),paste(round(pcltCompVal,3),"%",sep=""),ropeRad,paste(round(pcInROPE*100,3),"%",sep=""),paste(round(HDIinROPE*100,3),"%",sep="")),ncol=1,byrow=TRUE)
colnames(results) <- c("value")
rownames(results) <- c("Mean","Median","Mode","HDI min","HDI max","Null Value (NV)","Posterior above NV","Posterior below NV","Region of Practical Equivalence (ROPE)", "Posterior in ROPE", "HDI in Rope")
results <- as.table(results)

cat("Comparison of posterior and null value using Kruschke's HDI and ROPE method\n\nTable summary\n")
print(results)

if (HDIinROPE == 1) {
  cat("\n")
  cat(paste("Recommendation: Accept Null Hypothesis (H0) for null value = ",compVal,".\n\n",sep=""))
} else if (HDIinROPE <= 0) {
  cat("\n")
  cat(paste("Recommendation: Reject Null Hypothesis (H0) for null value = ",compVal,".\n\n",sep=""))
} else {
  cat("\n")
  cat(paste("Recommendation: Unclear Decision for Null Hypothesis (H0) with null value = ",compVal,". Some of the HDI lies in ROPE but not completely.\n\n",sep=""))
}

cat(paste("Recommended reporting: \n",sep=""))
cat(paste("M = ",round(meansample,3),", ",(HDImass*100),"% ","HDI ","[",round(HDImin,3),", ",round(HDImax,3),"], Null Value = ",compVal,", ROPE = ",ropeRad,", Posterior within ROPE = ",(round(100*pcInROPE,3)),"%, HDI within ROPE = ",(round(100*HDIinROPE,5)),"%",sep=""))
}
